using UnityEngine;

public class RealisticMissile : MonoBehaviour
{
    [Header("Raketen-Parameter")]
    public float damage = 100f;
    public float maxSpeed = 1000f;
    public float acceleration = 100f;
    public float turnSpeed = 5f;
    public float homingStrength = 10f;
    public float maxTurnAngle = 45f;
    public float lifeTime = 10f;
    public float explosionRadius = 10f;
    public GameObject explosionEffect;

    [Header("Ziel-Tag")]
    public string targetTag = "Player";

    private Transform target;
    private Rigidbody rb;
    private float timeAlive = 0f;
    private Vector3 velocity;
    private bool isTargetLocked = false;

    void Start()
    {
        rb = GetComponent<Rigidbody>();

        if (explosionEffect == null)
        {
            Debug.LogWarning("Kein Explosionseffekt zugewiesen!");
        }

        target = FindNewTarget();
        isTargetLocked = target != null;
    }

    void Update()
    {
        timeAlive += Time.deltaTime;

        // Explodieren nach Ablauf der Lebenszeit
        if (timeAlive > lifeTime)
        {
            Explode();
            return;
        }

        // Wenn Ziel weg oder zerstört, neues suchen
        if (target == null)
        {
            target = FindNewTarget();
            isTargetLocked = target != null;
        }

        // Ziel im Sichtfeld? Dann verfolgen
        if (target != null && IsInSight(target))
        {
            if (!isTargetLocked)
            {
                LockOnTarget();
            }

            Vector3 directionToTarget = (target.position - transform.position).normalized;
            float angleToTarget = Vector3.Angle(transform.forward, directionToTarget);

            if (angleToTarget < maxTurnAngle)
            {
                Vector3 newDirection = Vector3.RotateTowards(transform.forward, directionToTarget, turnSpeed * Time.deltaTime, 0f);
                transform.rotation = Quaternion.LookRotation(newDirection);
            }

            velocity = transform.forward * maxSpeed;
            rb.velocity = Vector3.Lerp(rb.velocity, velocity, acceleration * Time.deltaTime);
            transform.Translate(Vector3.forward * maxSpeed * Time.deltaTime);

            if (Vector3.Distance(transform.position, target.position) < explosionRadius)
            {
                Explode();
            }
        }
        else
        {
            // Kein Ziel oder nicht im Sichtfeld – geradeaus fliegen
            velocity = transform.forward * maxSpeed;
            rb.velocity = Vector3.Lerp(rb.velocity, velocity, acceleration * Time.deltaTime);
            transform.Translate(Vector3.forward * maxSpeed * Time.deltaTime);
        }
    }

    void LockOnTarget()
    {
        isTargetLocked = true;
    }

    Transform FindNewTarget()
    {
        GameObject[] targets = GameObject.FindGameObjectsWithTag(targetTag);
        Transform closest = null;
        float closestAngle = maxTurnAngle;

        foreach (GameObject obj in targets)
        {
            if (obj == null) continue;
            Transform t = obj.transform;
            if (IsInSight(t))
            {
                float angle = Vector3.Angle(transform.forward, (t.position - transform.position));
                if (angle < closestAngle)
                {
                    closest = t;
                    closestAngle = angle;
                }
            }
        }

        return closest;
    }

    bool IsInSight(Transform potentialTarget)
    {
        Vector3 dirToTarget = (potentialTarget.position - transform.position).normalized;
        float angle = Vector3.Angle(transform.forward, dirToTarget);
        return angle <= maxTurnAngle;
    }

    void Explode()
    {
        if (explosionEffect != null)
        {
            Instantiate(explosionEffect, transform.position, transform.rotation);
        }

        Collider[] hitColliders = Physics.OverlapSphere(transform.position, explosionRadius);
        foreach (Collider hit in hitColliders)
        {
            if (hit.CompareTag("Player") || hit.CompareTag("Enemy"))
            {
                Health targetHealth = hit.GetComponent<Health>();
                if (targetHealth != null)
                {
                    targetHealth.TakeDamage(damage);
                }
            }
        }

        Destroy(gameObject);
    }

    void OnDrawGizmos()
    {
        Gizmos.color = Color.red;
        Gizmos.DrawWireSphere(transform.position, explosionRadius);
    }
}
