using UnityEngine;

public class Spawner : MonoBehaviour
{
    [Header("reach")]
    public float activationRange = 10f;

    [Header("Gizmo settings")]
    public Color gizmoColor = Color.red;
    public bool showRangeGizmo = true;
    public bool showDirectionGizmo = true;
    public float gizmoDirectionLength = 2f;

    [Header("Spawn settings")]
    public GameObject objectToSpawn;         
    public float spawnDistance = 2f;         

    [Header("Fire frequency")]
    public float fireRate = 2f;              

    [Header("Field of view detection")]
    [Range(0f, 1f)]
    public float viewAngleThreshold = 0.5f; 

    [Header("Spawn direction")]
    public bool useFixedZDirection = false;  

    [Header("Ammo-System")]
    public bool isAmmoUnlimited = false;   
    public int maxAmmo = 10;               
    private int currentAmmo; 
    private float nextFireTime = 0f;
    private Transform player;
    private bool isSpawnerActive = false; 

    void Start()
    {
        
        currentAmmo = maxAmmo;

        if (Camera.main != null)
        {
            player = Camera.main.transform;
        }
    }

    void Update()
    {
        
        if (Vector3.Distance(transform.position, player.position) <= activationRange)
        {
            if (!isSpawnerActive)
            {
                isSpawnerActive = true;
                Debug.Log("Spawner aktiviert!");
            }

            
            if (isSpawnerActive && Time.time >= nextFireTime && IsSpawnerInView() && player != null)
            {
                if (!isAmmoUnlimited && currentAmmo <= 0)
                {
                    Debug.Log("Out of Ammo!");
                    return;
                }

                if (useFixedZDirection)
                    SpawnAlongZAxisTowardPlayer();
                else
                    SpawnDirectlyTowardPlayer();

                
                nextFireTime = Time.time + (1f / fireRate);

                
                if (!isAmmoUnlimited)
                {
                    currentAmmo--;
                    Debug.Log("Ammo Left: " + currentAmmo);
                }
            }
        }
        else
        {
            if (isSpawnerActive)
            {
                isSpawnerActive = false;
                Debug.Log("Spawner deaktiviert!");
            }
        }
    }

    private bool IsSpawnerInView()
    {
        if (player == null) return false;

        Vector3 toSpawner = (transform.position - player.position).normalized;
        float dot = Vector3.Dot(player.forward, toSpawner);

        return dot > viewAngleThreshold;
    }

    private void SpawnDirectlyTowardPlayer()
    {
        Vector3 directionToPlayer = (player.position - transform.position).normalized;
        Vector3 spawnPosition = transform.position + directionToPlayer * spawnDistance;
        Quaternion spawnRotation = Quaternion.LookRotation(directionToPlayer);

        Instantiate(objectToSpawn, spawnPosition, spawnRotation);
    }

    private void SpawnAlongZAxisTowardPlayer()
    {
        Vector3 toPlayer = player.position - transform.position;
        float dot = Vector3.Dot(transform.forward, toPlayer.normalized);

        Vector3 spawnDirection = (dot >= 0f) ? transform.forward : -transform.forward;
        Vector3 spawnPosition = transform.position + spawnDirection * spawnDistance;
        Quaternion spawnRotation = Quaternion.LookRotation(spawnDirection);

        Instantiate(objectToSpawn, spawnPosition, spawnRotation);
    }

    
    public void ReloadAmmo()
    {
        currentAmmo = maxAmmo;
        Debug.Log("Ammo Reloaded!");
    }

    
    void OnDrawGizmos()
    {
        if (showRangeGizmo)
        {
            Gizmos.color = gizmoColor;
            Gizmos.DrawWireSphere(transform.position, activationRange);
        }

        if (showDirectionGizmo)
        {
            Gizmos.color = Color.blue;
            Gizmos.DrawLine(transform.position, transform.position + transform.forward * gizmoDirectionLength);
        }
    }

}
